/** @file   controlevent.h
 * @brief   Declaration of ControlEvent - class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/05/15 19:18:36 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_CONTROLEVENT_H
#define H_WWW_CONTROLEVENT_H

#include "dynamicevent.h"
#include "vec2d.h"
#include "rect2d.h"

namespace WeWantWar {

enum MusicEventAction {
  EFadeOut,
  EFadeIn,
  ESetVolume,
  EChangeMusic,
  EStopMusic
};

/** @class  ControlEvent
 * @brief   Controls the game structures and sends messages to the GameObjects.
 * @author  Tomi Lamminsaari
 *
 * So far this event can do the following things:
 * - Lock the screen so that it does not scroll. When screen is locked to a
 *   certain position the player cannot cross the screen borders.
 * - Send messages to the instances of GameObject with certain id-number.
 * - Change the primary target of the aliens.
 * - Teleport objects from place to another.
 * - Force an object to capture another object.
 * - Control the background music
 */
class ControlEvent : public DynamicEvent
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Datatype controlcommands type. */
  typedef int ItemType;
  
  /** ID code of controlcommand that handles the screenlock */
  static const ItemType CTRL_SCREENLOCK = 0;
  /** ID code of controlcommand that sets the target objects of the enemies */
  static const ItemType CTRL_AI_TARGETS = 1;
  /** ID code of the controlcommand that sends a message a GameObject. */
  static const ItemType CTRL_OBJECT_MESSAGE = 2;
  /** ID code of the controlcommand that teleports an object to a new place */
  static const ItemType CTRL_OBJECT_TELEPORT = 3;
  /** ID code of the controlcommand that makes an object to capture another
   * object.
   */
  static const ItemType CTRL_OBJECT_CAPTURE = 4;
  /** ID code of the controlcommand that changes the background music. */
  static const ItemType CTRL_MUSIC_CONTROL = 5;
  
  /** The data of the control-events are stored in structures of this type.
   * The data we store to these fields depends on the type of the
   * item.
   *  - CTRL_SCREENLOCK event. The @c pos1 - field contains the coordinate of
   *    the topleft corner where the screen will be locked. @c pos2 - field
   *    holds the position where player will be translated. @c intData1 - field
   *    tells if the screenlock should be set on or off.
   *  - CTRL_AI_TARGETS event. The @c intData1 - field holds the ID-code of
   *    object that will be set as primary target to the aliens. -1 sets the
   *    player to be the primary target.
   *  - CTRL_OBJECT_MESSAGE event. Sends a message to certain GameObject.
   *  - CTRL_OBJECT_TELEPORT event. Teleports an object to another place.
   *    The @c pos1 - member should have the coordinate where to teleport.
   *  - CTRL_BACKGROUND_MUSIC event. The @c intData1 field holds the id code
   *    of the music action we should perform.
   */
  struct ControlItem {
    /** Type of this control-command. See datatypes.h - file for more
     * information.
     */
    ItemType type;
    
    /** Coordinate data. */
    eng2d::Vec2D pos1;
    /** Coordinate data */
    eng2d::Vec2D pos2;
    /** Coordinate data */
    eng2d::Vec2D pos3;
    /** A rectangle data */
    eng2d::Rect2D rect1;
    
    /** An integer data */
    int intData1;
    /** An integer data */
    int intData2;
    /** An integer data */
    int intData3;
    
    /** An id-code that unifies the target object. */
    int idcode;
  };


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	ControlEvent();


	/** Destructor
   */
	virtual ~ControlEvent();

private:

	/** Copy constructor.
   * @param     rO                Reference to another ControlEvent
   */
  ControlEvent( const ControlEvent& rO );

	/** Assignment operator
   * @param     rO                Reference to another ControlEvent
   * @return    Reference to us.
   */
  ControlEvent& operator = ( const ControlEvent& rO );

public:


  ///
  /// Methods
  /// =======

  /** Activates this event. The update() - method does nothing unless the
   * event has been activated.
   */
  virtual void activate();
  
  /** Reads the contents of this event from Level Description File. This method
   * expects that the opening CONTROL - tag has been read away from the stream.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  virtual int readData( std::istream& rIn );
  
  /** Updates this event. If this event hasn't been activated, this method
   * does nothing.
   */
  virtual void update();


  ///
  /// Getter methods
  /// ==============


protected:

  /** Reads the data about locking the screen.
   * @param     rIn               The input stream
   * @return    Pointer to ControlItem that holds the data or nullpointer if
   *            loading fails.
   */
  ControlItem* readScreenLock( std::istream& rIn );
  
  /** Reads the data about the changing the primary target.
   * @param     rIn               The input stream
   * @return    Pointer to ControlItem or nullpointer if loading failed.
   */
  ControlItem* readAITargets( std::istream& rIn );
  
  /** Reads the data about the controller-message
   * @param     rIn               The input stream
   * @return    Pointer to ControlItem or nullpointer if loading failed.
   */
  ControlItem* readObjectMessage( std::istream& rIn );
  
  /** Reads the data about the teleport - command
   * @param     rIn               The input-stream
   * @return    Pointer to ControlItem or nullpointer if loading failed
   */
  ControlItem* readObjectTeleport( std::istream& rIn );
  
  /** Reads the data about the capture - command
   * @param     rIn               The input stream
   * @return    Pointer to ControlItem or nullpointer if failed.
   */
  ControlItem* readObjectCapture( std::istream& rIn );
  
  /** Reads the data about the music control.
   * @param     rIn               The input stream
   * @return    Pointer to ControlItem or nullpointer if failed.
   */
  ControlItem* readMusicControl( std::istream& rIn );
  
  /** Activates the screen lock.
   */
  void launchEvent();
  
  /** Locks the screen or releases the screenlock
   * @param     pI                Pointer to ControlItem that holds the data.
   */
  void doScreenLock( ControlItem* pI );
  
  /** Changes the primary ai target
   * @param     pI                Pointer to ControlItem that holds the data
   */
  void doAiTargets( ControlItem* pI );
  
  /** Sends a message to a GameObject
   * @param     pI                Pointer to ControlItem that holds the data.
   */
  void doObjectMessage( ControlItem* pI );
  
  /** Teleports the object
   * @param     pI                Pointer to ControlItem that holds the data.
   */
  void doObjectTeleport( ControlItem* pI );
  
  /** Captures an object.
   * @param     pI                Pointer to ControlItem that holds the data.
   */
  void doObjectCapture( ControlItem* pO );
  
  /** Music action
   * @param     pI                Pointer to ControlItem that holds the data.
   */
  void doMusicControl( ControlItem* pO );
  
  ///
  /// Members
  /// =======

  /** The control-items are in this vector. */
  std::vector< ControlItem* > m_controlItems;


private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: controlevent.h,v $
 * Revision 1.2  2006/05/15 19:18:36  lamminsa
 * Control options for Mp3 playback added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:52+02  lamminsa
 * Initial revision
 *
 */
 
